#include "CSerialCommHelper.h"
#include <Process.h>
#include <assert.h>
#include <atlbase.h>
#include <iostream>

/// <summary>
/// This function is used to invalidates the handle when no longer needed.
/// </summary>
/// <param name="hHandle">Handle to invalidate</param>
void CSerialCommHelper::InvalidateHandle(HANDLE& hHandle)
{
	hHandle = INVALID_HANDLE_VALUE;
}

/// <summary>
/// This function tries to close the handle and then invalidates it.
/// </summary>
/// <param name="hHandle">Handle to be closed</param>
void CSerialCommHelper::CloseAndCleanHandle(HANDLE& hHandle)
{
	BOOL abRet = CloseHandle(hHandle);
	if (!abRet)
	{
		assert(0);
	}
	InvalidateHandle(hHandle);
}

/// <summary>
/// This function receives raw position data from read thread and parses each 
/// coordinate i.e. S,X,Y and Z into position structure and forwards it to the
/// subscriber.
/// </summary>
/// <param name="data">Raw mouse data string</param>
void CSerialCommHelper::SetPosition(char* arr)
{
	try {
		StPosition newPosition;
		int x, y;
		int newx, newy, newz;

		x = get_int(arr[7], arr[8]);
		y = get_int(arr[9], arr[10]);
		newx = x + y; // the data is sent rotated to emulate the Immersion Mouse, so unrotate it for diaplay
		newy = y - x;
		newz = get_int(arr[11], arr[12]); // Z is normal
		newPosition.X = newx;
		newPosition.Y = newy;
		newPosition.Z = newz;

		// Forward new position data to subscriber.
		if (dCallback != 0)
			dCallback(newPosition);
	}
	catch (...) {
		// Do nothing.
	}
}

int CSerialCommHelper::get_int(int a, int b) { // convert to int
	int temp = (a << 7) + b; // a are the upper 7 bits, b the lower
	if (temp > 8192) temp -= 16384; // large numbers are negitive
	return temp;
}

/// <summary>
/// CSerialCommHelper Constructor.
/// </summary>
CSerialCommHelper::CSerialCommHelper()
{
	m_streamingDataStarted = false;
	InvalidateHandle(m_hThreadTerm);
	InvalidateHandle(m_hThread);
	InvalidateHandle(m_hThreadStarted);
	InvalidateHandle(m_hCommPort);
	InvalidateHandle(m_hDataRx);
}

/// <summary>
/// CSerialCommHelper Desconstructor.
/// </summary>
CSerialCommHelper::~CSerialCommHelper()
{
}

int CSerialCommHelper::Read(char* inbuff, int inbuflen)
{
	DWORD bytes_read;
	if (!ReadFile(m_hCommPort, inbuff, inbuflen - 1, &bytes_read, NULL)) return 0;
	inbuff[bytes_read] = 0;
	return (int)bytes_read;
}

int CSerialCommHelper::Write(char* outbuff, int outbuflen)
{
	DWORD bytes_written = 0;
	if (!WriteFile(m_hCommPort, outbuff, outbuflen, &bytes_written, NULL)) return 0;
	return (int)bytes_written;
}

/// <summary>
/// Sets up initial parameters for the device such data/stop bits, partiy, flow control and 
/// timeouts.
/// </summary>
/// <returns>TRUE if successful, otherwise returns FALSE.</returns>
HRESULT CSerialCommHelper::Init(const char* szPortName, DWORD dwBaudRate, BYTE byParity, BYTE byStopBits, BYTE byByteSize)
{
	HRESULT hr = S_OK;
	try
	{
		size_t size = strlen(szPortName) + 1;
		wchar_t* portName = new wchar_t[size];

		size_t outSize;
		mbstowcs_s(&outSize, portName, size, szPortName, size - 1);

		m_hDataRx = CreateEvent(0, 0, 0, 0);

		m_hCommPort = ::CreateFile(portName,
			GENERIC_READ | GENERIC_WRITE,//access ( read and write)
			0,	//(share) 0:cannot share the COM port						
			0,	//security  (None)				
			OPEN_EXISTING,// creation : open_existing
			FILE_ATTRIBUTE_NORMAL,// we want normal operation
			0// no templates file for COM port...
		);
		if (m_hCommPort == INVALID_HANDLE_VALUE)
		{
			return E_FAIL;
		}

		COMMTIMEOUTS timeouts;
		COMMCONFIG dcbSerialParams;

		if (!GetCommState(m_hCommPort, &dcbSerialParams.dcb)) return E_FAIL;

		dcbSerialParams.dcb.DCBlength = sizeof(dcbSerialParams.dcb);
		dcbSerialParams.dcb.BaudRate = CBR_9600;
		dcbSerialParams.dcb.ByteSize = 8;
		dcbSerialParams.dcb.StopBits = ONESTOPBIT;
		dcbSerialParams.dcb.Parity = NOPARITY;

		dcbSerialParams.dcb.fBinary = TRUE;
		dcbSerialParams.dcb.fDtrControl = DTR_CONTROL_DISABLE;
		dcbSerialParams.dcb.fRtsControl = RTS_CONTROL_DISABLE;
		dcbSerialParams.dcb.fOutxCtsFlow = FALSE;
		dcbSerialParams.dcb.fOutxDsrFlow = FALSE;
		dcbSerialParams.dcb.fDsrSensitivity = FALSE;
		dcbSerialParams.dcb.fAbortOnError = TRUE;

		if (!SetCommState(m_hCommPort, &dcbSerialParams.dcb)) return E_FAIL;

		GetCommTimeouts(m_hCommPort, &timeouts);
		timeouts.ReadIntervalTimeout = 50;
		timeouts.ReadTotalTimeoutConstant = 50;
		timeouts.ReadTotalTimeoutMultiplier = 1;
		timeouts.WriteTotalTimeoutConstant = 50;
		timeouts.WriteTotalTimeoutMultiplier = 1;

		if (!SetCommTimeouts(m_hCommPort, &timeouts))return E_FAIL;

		m_abIsConnected = true;
	}
	catch (...)
	{
		assert(0);
		hr = E_FAIL;
	}
	return hr;
}

/// <summary>
/// This function sets state as Started.
/// </summary>
/// <returns></returns>
HRESULT CSerialCommHelper::Start()
{
	init_IMMC();

	Write((char*)"BEGIN", 5);
	char inbuff[4];
	int num = Read(inbuff, 4);
	//Sleep(1000);
	//std::cout << "Send streaming command now. " << std::endl;
	//char st[25] = { 0xCF, 0x00, 0x64, 0x0A, 0x0F, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01 };

	//Write(st, 25);
	//num = Read(inbuff, 1);

	//m_streamingDataStarted = true;*/

	m_hThreadStarted = CreateEvent(0, 0, 0, 0);
	assert(m_hThreadStarted && "Could not create event");

	m_hThread = (HANDLE)_beginthreadex(0, 0, CSerialCommHelper::ThreadFn, (void*)this, 0, 0);

	DWORD dwWait = WaitForSingleObject(m_hThreadStarted, INFINITE);
	assert(dwWait == WAIT_OBJECT_0);
	CloseHandle(m_hThreadStarted);
	InvalidateHandle(m_hThreadStarted);
	


	return S_OK;
}

/// <summary>
/// This function sets state as Stopped.
/// </summary>
/// <returns></returns>
HRESULT CSerialCommHelper::Stop()
{
	return S_OK;
}

/// <summary>
/// This function closes the read thread and removes the subscriber callback.
/// Its automatically called when Close() is called.
/// </summary>
/// <returns>Returns S_OK if successful, E_FAIL if fails.</returns>
HRESULT CSerialCommHelper::UnInit()
{
	HRESULT hr = S_OK;
	try
	{
		quit = true;
		m_streamingDataStarted = false;
		Write((char*)"E", 1);
		m_abIsConnected = false;
		SignalObjectAndWait(m_hThreadTerm, m_hThread, INFINITE, FALSE);
		CloseAndCleanHandle(m_hThreadTerm);
		CloseAndCleanHandle(m_hThread);
		CloseAndCleanHandle(m_hDataRx);
		CloseAndCleanHandle(m_hCommPort);
	}
	catch (...)
	{
		assert(0);
		hr = E_FAIL;
	}

	dCallback = nullptr;

	return hr;
}

/// <summary>
/// Its a thread function that runs in a loop and waits for a COMM event. If finds any new bytes to reads,
/// and forwards the string to SetPosition function.
/// </summary>
/// <param name="pvParam">CSerialCommHelper object to receive data updates through SetPosition function.</param>
/// <returns></returns>
unsigned __stdcall CSerialCommHelper::ThreadFn(void* pvParam)
{
	CSerialCommHelper* apThis = (CSerialCommHelper*)pvParam;
	bool abContinue = true;
	bool isComplete = true;
	DWORD dwEventMask = 0;

	DWORD RxBytes;
	DWORD BytesReceived;
	char RxBuffer[64] = { '\0' };
	DWORD dwWait;

	char array1[1] = { 0x0A }; // to convert byte to str
	
	DWORD bytes_read = 13;
	char array2[13] = { '\0' };
	DWORD bytes_written;

	SetEvent(apThis->m_hThreadStarted);

	while (abContinue && !apThis->quit)
	{
		try
		{
			
			std::chrono::steady_clock::time_point start;
			if (bytes_read == 13) {
				start = std::chrono::steady_clock::now();
			}

			WriteFile(apThis->m_hCommPort, array1, 1, &bytes_written, NULL); // request a data record
			//Sleep(1);
			BOOL abRet = ReadFile(apThis->m_hCommPort, array2, 13, &bytes_read, NULL);

			if (!abRet)
			{
				abContinue = FALSE;
				break;
			}

			if (bytes_read == 13) {
				apThis->SetPosition(array2);

				
				auto end = std::chrono::steady_clock::now();

				std::cout
					<< std::chrono::duration_cast<std::chrono::milliseconds>(end - start).count()
					<< " ms." << std::endl;
			};

			Sleep(10);
		}
		catch (...)
		{
		}
	}
	return 0;
}

void CSerialCommHelper::print_hex(std::string string)
{
	//unsigned char* p = (unsigned char*)string;

	for (int i = 0; i < string.length(); ++i) {
		if (!(i % 16) && i)
			printf("\n");

		printf("0x%02x ", string[i]);
	}
	printf("\n\n");
}


/// <summary>
/// This function is used to subscribe to the data updates.
/// </summary>
/// <param name="cB">Callback function of void return type and one input parameter of type StPosition</param>
void CSerialCommHelper::setDCallback(type_myCallBack cB)
{
	dCallback = cB;
}



/// <summary>
/// Reads registry keys for FTDIBUS and enumerates COM ports.
/// </summary>
/// <returns>Returns true if successful, returns false if fails.</returns>
bool CSerialCommHelper::detectPort()
{
	ports.clear();

	ATL::CRegKey FTDIBUSKey;
	LSTATUS nStatus{ FTDIBUSKey.Open(HKEY_LOCAL_MACHINE, _T("SYSTEM\\CurrentControlSet\\Enum\\FTDIBUS"), KEY_ENUMERATE_SUB_KEYS) };
	if (nStatus != ERROR_SUCCESS)
	{
		SetLastError(nStatus);
		return false;
	}

	TCHAR deviceSubkey[512];
	DWORD dskLength = 512;
	int i = 0;

	while (true)
	{
		nStatus = FTDIBUSKey.EnumKey(i, deviceSubkey, &dskLength);
		if (nStatus == ERROR_SUCCESS)
		{
			ATL::CAtlString RootAndDevice;
			RootAndDevice.Format(_T("SYSTEM\\CurrentControlSet\\Enum\\FTDIBUS\\%s"), deviceSubkey);

			ATL::CRegKey DeviceKey;
			nStatus = DeviceKey.Open(HKEY_LOCAL_MACHINE, RootAndDevice, KEY_ENUMERATE_SUB_KEYS);
			if (nStatus != ERROR_SUCCESS)
			{
				SetLastError(nStatus);
				return false;
			}

			TCHAR DIndexSubkey[512];
			DWORD diskLength = 512;
			int j = 0;

			while (true)
			{
				nStatus = DeviceKey.EnumKey(j, DIndexSubkey, &diskLength);
				if (nStatus == ERROR_SUCCESS)
				{
					ATL::CAtlString RootAndDeviceAndIndex;
					RootAndDeviceAndIndex.Format(_T("SYSTEM\\CurrentControlSet\\Enum\\FTDIBUS\\%s\\%s"), deviceSubkey, DIndexSubkey);

					ATL::CRegKey DeviceIndexKey;
					nStatus = DeviceIndexKey.Open(HKEY_LOCAL_MACHINE, RootAndDeviceAndIndex, KEY_ENUMERATE_SUB_KEYS);
					if (nStatus != ERROR_SUCCESS)
					{
						SetLastError(nStatus);
						return false;
					}

					TCHAR DeviceParamsSubkey[512];
					DWORD dpLength = 512;
					int k = 0;
					while (true)
					{
						nStatus = DeviceIndexKey.EnumKey(k, DeviceParamsSubkey, &dpLength);
						if (nStatus == ERROR_SUCCESS)
						{
							ATL::CAtlString DeviceParams;
							DeviceParams.Format(_T("SYSTEM\\CurrentControlSet\\Enum\\FTDIBUS\\%s\\%s\\%s"), deviceSubkey, DIndexSubkey, DeviceParamsSubkey);

							ATL::CRegKey DeviceParamsKey;
							nStatus = DeviceParamsKey.Open(HKEY_LOCAL_MACHINE, DeviceParams, KEY_QUERY_VALUE);
							if (nStatus != ERROR_SUCCESS)
							{
								SetLastError(nStatus);
								return false;
							}

							TCHAR szPath[MAX_PATH] = _T("");
							ULONG nPathChars = _countof(szPath);
							nStatus = DeviceParamsKey.QueryStringValue(_T("PortName"), szPath, &nPathChars);
							if (nStatus == ERROR_SUCCESS)
							{
								std::wstring ws(szPath);
								// As the port name will not have unicode characters. 
#pragma warning(suppress: 4244)
								std::string str(ws.begin(), ws.end());
								ports.push_back(str);
								return true;
							}
						}
						k++;
					}
				}
				j++;
			}
		}
		else
			break;
		i++;
	}

	return false;
}

int CSerialCommHelper::init_IMMC()
{
	std::cout << "IMMC" << std::endl;
	char outbuff[] = "IMMC";
	char cr[] = "\r";
	char inbuff[4];
	int i, num;

	for (i = 0; i < 20; i++)
	{
		Write(cr, strlen(cr)); // if sending cr returns cr, then the mouse is Z-Type
		num = Read(inbuff, 4); // so return -99
		if (strchr(inbuff, '\r')) return -99; // Z-Type returns Vxx<cr>, so we can test for V or cr
	}

	for (i = 0; i < 20; i++)
	{
		Write(outbuff, 4);
		num = Read(inbuff, 4);
		if (strchr(inbuff, 'C')) return 1;
	}
	return 0;
}

